
/* Copyright (C) 2001-2007 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* fs_autohint.c */

/* process hints in stik fonts, using the resurrected 'AA' opcode */
/* there are 3 flavors of autohinting: Latin, Asian, and other. */
/* they are differentiated by the argument to the 'AA' opcode. */


#include "fs_itype.h"

#ifdef FS_STIK

#define MAX_HINTS 32

#define F26DOT6_FLOOR(x) ((x)&~63)
#define F26DOT6_ROUND(x)  F26DOT6_FLOOR(32+(x))

/****************************************************************/
/************************** misc *******************************/
/****************************************************************/
/* return index of v in v[] ... return <n> if not found */
static int indexof(FS_LONG x, int n, FS_LONG *v)
{
    int i;

    for (i = 0; i < n && x != v[i]; i++) ;
    return i;
}

/****************************************************************/
/* val lies between two values of old[], return value having the
* same percentage relationship with corresponding members of newv[]
*
* that is solve the following equation for <result>
*
*  val - old_lo      result - new_lo
*  --------------- = ---------------
*  old_hi - old_lo   new_hi - new_lo
*/
static FS_FIXED interp(F26DOT6 val, int num, F26DOT6 *oldv, F26DOT6 *newv)
{
    int i;
    FS_FIXED a, b, c, d;

    if (num < 1) return 0;

    /* find largest <i> such that oldv[i] <= val */
    for (i = num - 1; i > 0 && oldv[i] > val; i--) ;

    if (val == oldv[i])
        return newv[i];

    a = newv[i + 1] - newv[i];
    b = val - oldv[i];
    c = oldv[i + 1] - oldv[i];
    d = newv[i] + LongMulDiv(a, b, c);
    return d;
}


/****************************************************************/
#ifdef GRID_DEBUG
FS_VOID dump_element(char *s, fnt_ElementType *p)
{
    FS_SHORT i, np, nc = p->nc;
    FS_USHORT *ep;
    F26DOT6 *x = p->x;
    F26DOT6 *ox = p->ox;
    F26DOT6 *y = p->y;
    F26DOT6 *oy = p->oy;

    if (nc)
    {
        ep = p->ep;
        np = ep[nc - 1] - 1;

        FS_PRINTF(("-----dump_element----- %s\n", s));

        FS_PRINTF(("nc=%d ep=[", nc));
        for (i = 0; i < nc; i++)
            FS_PRINTF(("%d ", ep[i]));
        FS_PRINTF(("]\n"));

        FS_PRINTF(("-i- ---x--- ---x--- ---y--- ---y---    ---ox-- --ox--- --oy--- --oy---\n"));

        for (i = 0; i < np; i++)
            FS_PRINTF(("%3d %7ld %7.3f %7ld %7.3f    %7ld %7.3f %7ld %7.3f\n", i,
                       x[i], x[i] / 64.0, y[i], y[i] / 64.0,
                       ox[i], ox[i] / 64.0, oy[i], oy[i] / 64.0));

        FS_PRINTF(("phantom's\n"));
        np += PHANTOMCOUNT;
        for (/**/; i < np; i++)
            FS_PRINTF(("%3d %7ld %7.3f %7ld %7.3f    %7ld %7.3f %7ld %7.3f\n", i,
                       x[i], x[i] / 64.0, y[i], y[i] / 64.0,
                       ox[i], ox[i] / 64.0, oy[i], oy[i] / 64.0));
        FS_PRINTF(("\n"));
    }
}
#endif /* GRID_DEBUG */
/****************************************************************/

#ifdef GRID_DEBUG
FS_VOID dump_hints(int n, F26DOT6 *oldv, F26DOT6 *newv)
{
    int i;

    FS_PRINTF(("--hints--\n"));
    for (i = 0; i < n; i++)
        FS_PRINTF(("%d %f %f\n", i, oldv[i] / 64.0, newv[i] / 64.0));
}
#endif /* GRID_DEBUG */

/****************************************************************/
/* shell sort oldv[] and newv[] in ascending order by oldv[] */
static FS_VOID sort_hints(int n, F26DOT6 *oldv, F26DOT6 *newv)
{
    int gap, i, j;
    FS_FIXED temp;

    for (gap = n / 2; gap > 0; gap /= 2)
    {
        for (i = gap; i < n; i++)
        {
            for (j = i - gap; j >= 0 && oldv[j] > oldv[j + gap]; j -= gap)
            {
                temp = oldv[j];
                oldv[j] = oldv[j + gap];
                oldv[j + gap] = temp;
                temp = newv[j];
                newv[j] = newv[j + gap];
                newv[j + gap] = temp;
            }
        }
    }
#ifdef GRID_DEBUG
    dump_hints(n, oldv, newv);
#endif /* GRID_DEBUG */
}
/****************************************************************/
/* possibly add <oldv,new> to <oldv_v[],new_v[] */
static FS_SHORT add_pair(F26DOT6 oldv, F26DOT6 newv, FS_SHORT num, F26DOT6 *oldv_v, F26DOT6 *new_v)
{
    /* ? not already there */
    if (num == indexof(oldv, num, oldv_v))
    {
        if (num < MAX_HINTS)
        {
            oldv_v[num] = oldv;
            new_v[num] = newv;
            num++;
        }
    }
    return num;
}

/****************************************************************/
/* round coord <z> to grid or half grid */
static F26DOT6 odd_round(F26DOT6 z, int odd)
{
    if (odd)
    {
        /* half grid (1.5, 2.5, 3.5, 4.5, etc..) */
        z = 0x00000020 + (z & 0xFFFFFF40);
    }
    else
    {
        /* grid (1, 2, 3, 4, etc..) */
        z = (z + 0x00000020) & 0xFFFFFF40;
    }
    return z;
}

/****************************************************************/
/* round coord <z> to grid or half grid */
static F26DOT6 rtg_rthg(F26DOT6 z, FS_SHORT sw)
{
    if (sw & 1)    /* odd ... half grid (1.5, 2.5, 3.5, 4.5, etc..) */
        z = 32 + (z & ~63);
    else        /* even ... grid (1, 2, 3, 4, etc..) */
        z = (z + 32) & ~63;
    return z;
}

/****************************************************************/
/* adjust coord<z> so that the closer side of the expanded stroke is on the grid line  */
/* adjust z to cover the closer grid line */
static F26DOT6 atcg(F26DOT6 z, F26DOT6 sw)
{
    F26DOT6 right, left, dl, dr, dis;
    right = z + sw / 2;
    left = z - sw / 2;
    dl = F26DOT6_ROUND(left) - left;
    dr = F26DOT6_ROUND(right) - right;
    if (ABS(dl) < ABS(dr))
        dis = dl;
    else
        dis = dr;

    return z + dis;
}

/****************************************************************/
/* is point <j> a contour startpoint or endpoint ? */
static int is_endpoint(int j, int nc, FS_USHORT *ep)
{
    int i;

    if (j == 0 || j == ep[0])
        return 1;

    for (i = 1; i < nc; i++)
        if (j == ep[i] || j == (ep[i - 1] + 1))
            return 1;
    return 0;
}

/******************** GENERIC ***********************************/
/****************************************************************/
/* align extrema, horizontal or vertical places with the grid */
static FS_VOID autohint_other_sub(FS_SHORT n_pts, F26DOT6 sw, F26DOT6 *v, FS_SHORT nc, FS_USHORT *ep, FS_BOOLEAN int_sw)
{
    F26DOT6 old_v[MAX_HINTS];
    F26DOT6 new_v[MAX_HINTS];
    F26DOT6 z;
    FS_SHORT i, num;
    F26DOT6 lo, hi;
    FS_SHORT i_sw = 0;

    if (int_sw) /* using integer sw */
    {
        i_sw  = (FS_SHORT)(sw >> 6);
        i_sw = MAX(1, i_sw);
    }
    lo = hi = v[0];
    for (i = 1; i < n_pts; i++)
    {
        if (v[i] > hi)
            hi = v[i];
        if (v[i] < lo)
            lo = v[i];
    }

    for (num = 0, i = 0; i < n_pts; i++)
    {
        /* select the interesting points */
        if (i + 1 < n_pts && v[i] == v[i + 1])
        {
            /* add a new "interesting point" to hints */
            if (int_sw)
                z = rtg_rthg(v[i], i_sw);
            else
                z = atcg(v[i], sw);
            num = add_pair(v[i], z, num, old_v, new_v);
        }
    }

    /* (maybe) add lo, hi */
    if (int_sw)
    {
        num = add_pair(lo, rtg_rthg(lo, i_sw), num, old_v, new_v);
        num = add_pair(hi, rtg_rthg(hi, i_sw), num, old_v, new_v);
    }
    else
    {
        num = add_pair(lo, atcg(lo, sw), num, old_v, new_v);
        num = add_pair(hi, atcg(hi, sw), num, old_v, new_v);
    }

    /* now interpolate other points wrt these */
    sort_hints(num, old_v, new_v);
    for (i = 0; i < n_pts; i++)
        v[i] = interp(v[i], num, old_v, new_v);

    /* finally round stroke endpoints to grid */
    for (i = 0; i < n_pts; i++)
    {
        if (is_endpoint(i, nc, ep))
        {
            if (int_sw)
                v[i] = rtg_rthg(v[i], i_sw);
            else
                v[i] = atcg(v[i], sw);
        }
    }
}

/****************************************************************/
FS_VOID autohint_other(fnt_LocalGraphicStateType *gs, FS_BOOLEAN int_sw)
{
    FS_SHORT nc, n_pts;
    FS_USHORT *ep;
    F26DOT6 *x = gs->CE0->x;
    F26DOT6 *y = gs->CE0->y;
    F26DOT6 xw, yw;
    FS_ULONG *store = (FS_ULONG *) gs->globalGS->store;

    /* no contours or under 2 points -- nothing to do */
    nc = gs->CE0->nc;
    n_pts = nc ? gs->CE0->ep[nc - 1] + 1 : 0;
    if (nc == 0 || n_pts == 0)
        return;
    ep = gs->CE0->ep;

    /* fetch stroke widths */
    xw = store[0];
    yw = store[1];

#ifdef GRID_DEBUG
    FS_PRINTF(("autohint_other: xw=%ld yw=%ld\n", xw, yw));
    dump_element("before hints", gs->CE0);
#endif /* GRID_DEBUG */

    autohint_other_sub(n_pts, xw, x, nc, ep, int_sw);
    autohint_other_sub(n_pts, yw, y, nc, ep, int_sw);

#ifdef GRID_DEBUG
    dump_element("after hints", gs->CE0);
#endif /* GRID_DEBUG */
}


/****************************************************************/
/********************* LATIN ************************************/
/****************************************************************/
/*
* try and get improved spacing by looking at each character between
* lowercase o's.  on the left we have o's RSB and the char's LSB,
* on the right we have the chars RSB and o's LSB.  Add the scaled
* font units and subtract the integer lsb/rsb for the 'o'. We may
* also change the char's advance width in the process.
*
* FYI the algorithm does not improve characters where lo_x==hi_x
* such as a simple 'i' ... have to deal with these manually .
*/

static FS_VOID autohint_latin_x(fnt_LocalGraphicStateType *gs, FS_USHORT n_pts, F26DOT6 xw, F26DOT6 *x, FS_BOOLEAN int_sw)
{
    FS_SHORT i;
    F26DOT6 lo_x, hi_x;
    FS_SHORT nc;
    FS_USHORT *ep;
    FS_SHORT i_xw;
    int base = gs->globalGS->cvtCount - NS_CVT_USED;
    nc = gs->CE0->nc;
    ep = gs->CE0->ep;

    /* find limits */
    lo_x = hi_x = x[0];

    for (i = 1; i < n_pts; i++)
    {
        if (x[i] > hi_x)
            hi_x = x[i];
        if (x[i] < lo_x)
            lo_x = x[i];
    }

    /* adjust side bearings ? */
    {
        F26DOT6 o_rsb, lsb, left, rsb, o_lsb, right, new_w, old_w;

        o_rsb = gs->GetCVTEntry(gs, base + NS_O_RSB);
        lsb = lo_x;
        left = F26DOT6_ROUND(o_rsb + lsb);

        /* unhinted right space */
        rsb = x[n_pts + RIGHTSIDEBEARING] - hi_x;
        o_lsb = gs->GetCVTEntry(gs, base + NS_O_LSB);

        right = F26DOT6_ROUND(rsb + o_lsb);

        /* side bearings for 'o' rounded to appropriate grid */
        old_w = F26DOT6_ROUND(o_lsb + o_rsb);
        if (int_sw)
        {
            i_xw = (FS_SHORT)(xw >> 6);
            i_xw = MAX(1, i_xw);
            o_lsb = rtg_rthg(o_lsb, i_xw);
        }
        else
            o_lsb = atcg(o_lsb, xw);
        o_rsb = old_w - o_lsb;

        /* subtraction yields grd aligned side bearings for current character */
        lsb = left - o_rsb;
        rsb = right - o_lsb;

        /* move character to zero */
        for (i = 0; i < n_pts; i++)
            x[i] -= lo_x;

        /* make black width of character integral */
        old_w = hi_x - lo_x;
        new_w = F26DOT6_ROUND(old_w);
        if (old_w)
        {
            for (i = 0; i < n_pts; i++)
                x[i] = LongMulDiv(x[i], new_w, old_w);
        }

        /* add back the (grid aligned) x-minimum */
        if (lsb)
        {
            for (i = 0; i < n_pts; i++)
                x[i] += lsb;
        }

        /* we may have changed the advance width, record it */
        /* note this is integer, since new_w is integer and both */
        /* of the side bearings are on the same (whole or half) grid */
        x[n_pts + RIGHTSIDEBEARING] = lsb + new_w + rsb;
    }

    /* now the Latin width correction based on stroke width */
    /* above 3% stroke width we expand by 1/2 the stroke width */
    /* below 3% we contract by 1/2 the stroke width */
    {
        FS_FIXED adj;
        SFNT *sfnt = gs->globalGS->sfnt;

        if (sfnt)
        {
            FS_LONG err;
            FS_FIXED xppm, yppm, tan_s;
            err = get_scale_inputs(sfnt->user_scale, &xppm, &yppm, &tan_s);
            if (err)
            {
                return;
            }
            adj = (yppm >> 16) * (sfnt->stroke_pct - FS_NORMAL_STROKE_PCT);
            adj >>= 10;    /* as 26.6 */
            adj >>= 1;    /* 50% */
            adj = F26DOT6_ROUND(adj);

            x[n_pts + RIGHTSIDEBEARING] += adj;
        }
    }

    /* now move interesting points to the grid, interpolate the rest */
    autohint_other_sub(n_pts, xw, x, nc, ep, int_sw);

}
/****************************************************************/
static FS_VOID autohint_latin_y(fnt_LocalGraphicStateType *gs,  FS_SHORT n_pts, F26DOT6 yw, F26DOT6 *y, FS_BOOLEAN int_sw)
{
    F26DOT6 br, bs, xr, xs, cr, cs;
    F26DOT6 gbr, gbs, gxr, gxs, gcr, gcs;
    F26DOT6 fuzz = 4;  /* 1/16-th pixel */
    F26DOT6 old_v[MAX_HINTS] = { 0 }, new_v[MAX_HINTS] = { 0 };
    F26DOT6 lo_y, hi_y;
    FS_SHORT i, num;
    FS_SHORT i_yw = 0;
    fnt_GlobalGraphicStateType *globalGS = gs->globalGS;
    int base = globalGS->cvtCount - NS_CVT_USED;
    if (int_sw)
    {
        i_yw = (FS_SHORT)(yw >> 6);
        i_yw = MAX(1, i_yw);
    }

    /* find extrema */
    lo_y = hi_y = y[0];
    for (i = 1; i < n_pts; i++)
    {
        if (y[i] > hi_y)
            hi_y = y[i];
        if (y[i] < lo_y)
            lo_y = y[i];
    }

    /* y_heights (in 26.6) extracted from font characters 'HOxo' */
    cs = gs->GetCVTEntry(gs, base + NS_CAP_SQUARE);
    cr = gs->GetCVTEntry(gs, base + NS_CAP_ROUND);
    xs = gs->GetCVTEntry(gs, base + NS_X_SQUARE);
    xr = gs->GetCVTEntry(gs, base + NS_X_ROUND);
    bs = gs->GetCVTEntry(gs, base + NS_BASE_SQUARE);
    br = gs->GetCVTEntry(gs, base + NS_BASE_ROUND);

    /* now move the square heights to the whole/half grid for integer sw or
       adjust them to cover the closer grid line for fractional sw */
    /* then move corresponding rounds an integer pixel away */
    /* note we truncate, rather than round to keep the */
    /* rounds equal to the squares until the difference is */
    /* greater than or equal to 1 pixel */
    if (int_sw)
    {
        gcs = rtg_rthg(cs, i_yw);
        gxs = rtg_rthg(xs, i_yw);
        gbs = rtg_rthg(bs, i_yw);
    }
    else
    {
        gcs = atcg(cs, yw);
        gxs = atcg(xs, yw);
        gbs = atcg(bs, yw);
    }

    gcr = gcs + F26DOT6_FLOOR(cr - cs);
    gxr = gxs + F26DOT6_FLOOR(xr - xs);
    gbr = gbs - F26DOT6_FLOOR(bs - br); /* yes, baseline round is below baseline square */

    /* reduce x-height -- gotta be room for ij dot and space */
    if (gcs - gxs <= 64)
        gxs -= 64;
    if (gcr - gxr <= 64)
        gxr -= 64;

    /* move key points wrt these zones */
    for (num = 0, i = 0; i < n_pts; i++)
    {
        /* interesting place? */
        if ((i + 1 < n_pts && y[i] == y[i + 1]) || y[i] == lo_y || y[i] == hi_y)
        {
            /* in baseline zone */
            if (br - fuzz <= y[i] && y[i] <= bs + fuzz)
            {
                /* within 1/2 pixel of square - snap to square */
                if (ABS(y[i] - bs) < 32)
                    num = add_pair(y[i], gbs, num, old_v, new_v);
                /* within 1/2 pixel of round - snap to round */
                else if (ABS(y[i] - br) < 32)
                    num = add_pair(y[i], gbr, num, old_v, new_v);
                else    /* just round to grid */
                {
                    if (int_sw)
                        num = add_pair(y[i], rtg_rthg(y[i], i_yw), num, old_v, new_v);
                    else
                        num = add_pair(y[i], atcg(y[i], yw), num, old_v, new_v);
                }
            }
            /* in cap-height zone */
            else if (cs - fuzz <= y[i] && y[i] <= cr + fuzz)
            {
                /* within 1/2 pixel of square - snap to square */
                if (ABS(y[i] - cs) < 32)
                    num = add_pair(y[i], gcs, num, old_v, new_v);
                /* within 1/2 pixel of round - snap to round */
                else if (ABS(y[i] - cr) < 32)
                    num = add_pair(y[i], gcr, num, old_v, new_v);
                else    /* just round to grid */
                {
                    if (int_sw)
                        num = add_pair(y[i], rtg_rthg(y[i], i_yw), num, old_v, new_v);
                    else
                        num = add_pair(y[i], atcg(y[i], yw), num, old_v, new_v);
                }
            }
            /* in x-height xone */
            else if (xs - fuzz <= y[i] && y[i] <= xr + fuzz)
            {
                /* within 1/2 pixel of square - snap to square */
                if (ABS(y[i] - xs) < 32)
                    num = add_pair(y[i], gxs, num, old_v, new_v);
                /* within 1/2 pixel of round - snap to round */
                else if (ABS(y[i] - xr) < 32)
                    num = add_pair(y[i], gxr, num, old_v, new_v);
                else    /* just round to grid */
                    if (int_sw)
                        num = add_pair(y[i], rtg_rthg(y[i], i_yw), num, old_v, new_v);
                    else
                        num = add_pair(y[i], atcg(y[i], yw), num, old_v, new_v);
            }
            else  /* no zone just an interesting place ?*/
            {
                if (int_sw)
                    num = add_pair(y[i], rtg_rthg(y[i], i_yw), num, old_v, new_v);
                else
                    num = add_pair(y[i], atcg(y[i], yw), num, old_v, new_v);
            }
        }
    }

    /* now interpolate other points wrt these */
    sort_hints(num, old_v, new_v);
    for (i = 0; i < n_pts; i++)
    {
        y[i] = interp(y[i], num, old_v, new_v);
    }
}


/****************************************************************/
FS_VOID autohint_latin(fnt_LocalGraphicStateType *gs, FS_BOOLEAN int_sw)
{
    FS_USHORT *sp = gs->CE0->sp;
    FS_USHORT *ep = gs->CE0->ep;
    F26DOT6 *x = gs->CE0->x;
    F26DOT6 *y = gs->CE0->y;
    FS_SHORT i, nc, np;
    F26DOT6 xw, yw;
    FS_SHORT i_xw, i_yw;
    FS_LONG *store = gs->globalGS->store;

    /* no contours or under 2 points */
    nc = gs->CE0->nc;
    np = nc ? gs->CE0->ep[nc - 1] + 1 : 0;
    if (nc == 0 || np < 2)
    {
        /* round the advance width */
        x[np + RIGHTSIDEBEARING] = F26DOT6_ROUND(x[np + RIGHTSIDEBEARING]);
        return;
    }

    /* fetch stroke widths */
    xw = store[0];
    yw = store[1];
    autohint_latin_x(gs, np, xw, x, int_sw);
    autohint_latin_y(gs, np, yw, y, int_sw);
    /* special case small two point contours (dots). If x coordinates
    * are the same and y coordinates are close, make them the same
    * and vice versa.
    */
    for (i = 0; i < nc; i++)
    {
        if (ep[i] == sp[i] + 1)
        {
            if (x[sp[i]] == x[ep[i]] && ABS(y[sp[i]] - y[ep[i]]) < 32)
                y[sp[i]] = y[ep[i]] = MAX(y[sp[i]], y[ep[i]]);
            if (y[sp[i]] == y[ep[i]] && ABS(x[sp[i]] - x[ep[i]]) < 32)
                x[sp[i]] = x[ep[i]] = MAX(x[sp[i]], x[ep[i]]);
        }
    }

    i_xw = (FS_SHORT)(xw >> 6);
    i_yw = (FS_SHORT)(xw >> 6);
    i_xw = MAX(1, i_xw);
    i_yw = MAX(1, i_yw);

    /* finally, rtg or atcg stroke endpoints (except start/stop of a closed contour) */
    /* rtg or atcg using integer or fractional sw */
    for (i = 0; i < nc; i++)
    {
        /* ? closed contour */
        if (x[sp[i]] == x[ep[i]] && y[sp[i]] == y[ep[i]])
            continue;

        if (int_sw)
        {
            x[sp[i]] = rtg_rthg(x[sp[i]], i_xw);
            y[sp[i]] = rtg_rthg(y[sp[i]], i_yw);
            x[ep[i]] = rtg_rthg(x[ep[i]], i_xw);
            y[ep[i]] = rtg_rthg(y[ep[i]], i_yw);
        }
        else
        {
            x[sp[i]] = atcg(x[sp[i]], xw);
            y[sp[i]] = atcg(y[sp[i]], yw);
            x[ep[i]] = atcg(x[ep[i]], xw);
            y[ep[i]] = atcg(y[ep[i]], yw);
        }
    }
}


/****************************************************************/
/************ Monospace GREEK, CYRILLIC, Latin ******************/
/****************************************************************/
/* do reference line adjustments in Y
   do NOT do the spacing adjustments in X */
FS_VOID autohint_mono(fnt_LocalGraphicStateType *gs, FS_BOOLEAN int_sw)
{
    FS_SHORT i, nc, n_pts;
    FS_USHORT *ep, *sp;
    F26DOT6 *x = gs->CE0->x;
    F26DOT6 *y = gs->CE0->y;
    F26DOT6 xw, yw;
    FS_SHORT i_xw = 0, i_yw = 0;
    FS_ULONG *store = (FS_ULONG *) gs->globalGS->store;

    /* no contours or under 2 points -- nothing to do */
    nc = gs->CE0->nc;
    n_pts = nc ? gs->CE0->ep[nc - 1] + 1 : 0;
    if (nc == 0 || n_pts == 0)
        return;
    ep = gs->CE0->ep;
    sp = gs->CE0->sp;

    /* fetch stroke widths */
    xw = store[0];
    yw = store[1];

#ifdef GRID_DEBUG
    FS_PRINTF(("autohint_other: xw=%ld yw=%ld\n", xw, yw));
    dump_element("before hints", gs->CE0);
#endif /* GRID_DEBUG */

    /* autohinting with integer or fractional sw */
    autohint_other_sub(n_pts, xw, x, nc, ep, int_sw);
    autohint_latin_y(gs, n_pts, yw, y, int_sw);

    /* special case small two point contours (dots). If x coordinates
    * are the same and y coordinates are close, make them the same
    * and vice versa.
    */
    for (i = 0; i < nc; i++)
    {
        if (ep[i] == sp[i] + 1)
        {
            if (x[sp[i]] == x[ep[i]] && ABS(y[sp[i]] - y[ep[i]]) < 32)
                y[sp[i]] = y[ep[i]] = MAX(y[sp[i]], y[ep[i]]);
            if (y[sp[i]] == y[ep[i]] && ABS(x[sp[i]] - x[ep[i]]) < 32)
                x[sp[i]] = x[ep[i]] = MAX(x[sp[i]], x[ep[i]]);
        }
    }

    if (int_sw) /* using integer sw */
    {
        i_xw = (FS_SHORT)(xw >> 6);
        i_yw = (FS_SHORT)(xw >> 6);
        i_xw = MAX(1, i_xw);
        i_yw = MAX(1, i_yw);
    }

    /* finally, rtg or atcg stroke endpoints (except start/stop of a closed contour) */
    for (i = 0; i < nc; i++)
    {
        /* ? closed contour */
        if (x[sp[i]] == x[ep[i]] && y[sp[i]] == y[ep[i]])
            continue;

        if (int_sw) /* integer sw, rtg or rthg */
        {
            x[sp[i]] = rtg_rthg(x[sp[i]], i_xw);
            y[sp[i]] = rtg_rthg(y[sp[i]], i_yw);
            x[ep[i]] = rtg_rthg(x[ep[i]], i_xw);
            y[ep[i]] = rtg_rthg(y[ep[i]], i_yw);
        }
        else /* fractional sw, round to the nearest integer */
        {
            x[sp[i]] = atcg(x[sp[i]], xw);
            y[sp[i]] = atcg(y[sp[i]], yw);
            x[ep[i]] = atcg(x[ep[i]], xw);
            y[ep[i]] = atcg(y[ep[i]], yw);
        }
    }
}

static FS_VOID set_bbox_range(FS_SHORT n_pts, F26DOT6 *y,
                              F26DOT6 *lo_y, F26DOT6 *hi_y)
{
    FS_SHORT i;
    *lo_y = *hi_y = y[0];
    for (i = 1; i < n_pts; i++)
    {
        if (y[i] > *hi_y)
            *hi_y = y[i];
        if (y[i] < *lo_y)
            *lo_y = y[i];
    }
}

/****************************************************************/
/*********************** ARABIC *********************************/
/****************************************************************/

FS_VOID autohint_arabic(fnt_LocalGraphicStateType *gs, FS_BOOLEAN int_sw)
{
    FS_LONG i;
    FS_SHORT nc, n_pts;
    SFNT *sfnt;
    int base = gs->globalGS->cvtCount - NS_CVT_USED - ARABIC_CVT_USED;
    F26DOT6 old_v[MAX_HINTS], new_v[MAX_HINTS];
    F26DOT6 lo_x, lo_y, hi_x, hi_y;
    F26DOT6 fuzz = 4; /* 1/16th pixel in 26.6 */
    F26DOT6 uab, guab;
    F26DOT6 umab, gumab;
    F26DOT6 lmab, glmab;
    F26DOT6 unt, gunt;
    F26DOT6 unb, gunb;
    F26DOT6 lnt, glnt;
    F26DOT6 lnb, glnb;
    F26DOT6 uht, guht;
    F26DOT6 uhb, guhb;
    F26DOT6 lhb, glhb;
    F26DOT6 lmhb, glmhb;
    F26DOT6 lmsb, glmsb;
    F26DOT6 lshb, glshb;
    F26DOT6 lssb, glssb;
    F26DOT6 cht, gcht;
    F26DOT6 cst, gcst;
    F26DOT6 stroke;
    F26DOT6 x_sw, y_sw;
    F26DOT6 bold_adj;
    FS_LONG x_odd, y_odd;

    FS_SHORT num;
    F26DOT6 *x = gs->CE0->x;
    F26DOT6 *y = gs->CE0->y;
    FS_BOOLEAN vanilla;
    FS_FIXED xppm, yppm, tan_s;
    FS_LONG error;

    sfnt = gs->globalGS->sfnt;
    if (!sfnt)
        return;

    vanilla = (sfnt->user_scale[0] > 0 && sfnt->user_scale[1] == 0 && sfnt->user_scale[2] == 0 && sfnt->user_scale[3] > 0);
    if (!vanilla)
        return;

    if (base < 0)
        return;

    /* no contours or under 2 points -- nothing to do */
    nc = gs->CE0->nc;
    n_pts = nc ? gs->CE0->ep[nc - 1] + 1 : 0;
    if (nc == 0 || n_pts == 0)
        return;

    error = get_scale_inputs(sfnt->user_scale, &xppm, &yppm, &tan_s);
    if (error)
        return;

    stroke = (FS_ROUND(yppm) * sfnt->stroke_pct) >> 10;
    bold_adj = (FS_ROUND(yppm) * 0x7ae) >> 10;
    if (int_sw) /* round only for integer stroke width */
    {
        if (stroke < 64 /* F26DOT6_ONE */)
            stroke = 64;
        stroke = F26DOT6_ROUND(stroke);

        if (bold_adj < 64 /* F26DOT6 ONE */)
            bold_adj = 64;
        bold_adj = F26DOT6_ROUND(bold_adj);
    }

    x_sw = y_sw = stroke;

    /* get the difference between the current stroke and the stroke if no
       stroke pct is used to keep bold and normal the same height at the same size */
    bold_adj = (stroke - bold_adj) >> 1;

    x_odd = F26DOT6_FLOOR((stroke + 32));
    x_odd = x_odd & 1;
    y_odd = x_odd;

    /* get CVT values in 26.6 for Arabic autohinting */
    cht = gs->GetCVTEntry(gs, base + ARABIC_CENTER_HIGH_TOP);
    cst = gs->GetCVTEntry(gs, base + ARABIC_CENTER_SHORT_TOP);
    uab = gs->GetCVTEntry(gs, base + ARABIC_UPPER_ACCENT_BOT);
    unt = gs->GetCVTEntry(gs, base + ARABIC_UPPER_NUMERAL_TOP);
    unb = gs->GetCVTEntry(gs, base + ARABIC_UPPER_NUMERAL_BOT);
    lnt = gs->GetCVTEntry(gs, base + ARABIC_LOWER_NUMERAL_TOP);
    lnb = gs->GetCVTEntry(gs, base + ARABIC_LOWER_NUMERAL_BOT);
    uht = gs->GetCVTEntry(gs, base + ARABIC_UPPER_HIGH_TOP);
    uhb = gs->GetCVTEntry(gs, base + ARABIC_UPPER_HIGH_BOT);
    lhb = gs->GetCVTEntry(gs, base + ARABIC_LOWER_HIGH_BOT);
    lmsb = gs->GetCVTEntry(gs, base + ARABIC_LOWER_MEDIUM_SHORT_BOT);
    lmhb = gs->GetCVTEntry(gs, base + ARABIC_LOWER_MEDIUM_HIGH_BOT);
    lssb = gs->GetCVTEntry(gs, base + ARABIC_LOWER_SHORT_SHORT_BOT);
    lshb = gs->GetCVTEntry(gs, base + ARABIC_LOWER_SHORT_HIGH_BOT);
    lmab = gs->GetCVTEntry(gs, base + ARABIC_LOWER_MEDIUM_ACCENT_BOT);
    umab = gs->GetCVTEntry(gs, base + ARABIC_UPPER_MEDIUM_ACCENT_BOT);

    if (int_sw)
    {
        glnt = odd_round(lnt - bold_adj, y_odd);
        glnb = odd_round(lnb + bold_adj, y_odd);
        gcst = odd_round(cst - bold_adj, y_odd);
        guht = odd_round(uht - bold_adj, y_odd);
        guhb = odd_round(uhb - bold_adj, y_odd);
        glhb = odd_round(lhb + bold_adj, y_odd);
        glmsb = odd_round(lmsb + bold_adj, y_odd);
        glssb = odd_round(lssb + bold_adj, y_odd);
        guab = odd_round(uab - bold_adj, y_odd);
        glmab = odd_round(lmab + bold_adj, y_odd);
        gumab = odd_round(umab - bold_adj, y_odd);
    }
    else
    {
        glnt = atcg(lnt - bold_adj, y_sw);
        glnb = atcg(lnb + bold_adj, y_sw);
        gcst = atcg(cst - bold_adj, y_sw);
        guht = atcg(uht - bold_adj, y_sw);
        guhb = atcg(uhb - bold_adj, y_sw);
        glhb = atcg(lhb + bold_adj, y_sw);
        glmsb = atcg(lmsb + bold_adj, y_sw);
        glssb = atcg(lssb + bold_adj, y_sw);
        guab = atcg(uab - bold_adj, y_sw);
        glmab = atcg(lmab + bold_adj, y_sw);
        gumab = atcg(umab - bold_adj, y_sw);
    }

    gunt = glnt + (0xFFFFFF40 & (unt - lnt));
    gunb = glnb - (0xFFFFFF40 & (lnb - unb));
    gcht = gcst + (0xFFFFFF40 & (cht - cst));
    glmhb  = glmsb - (0xFFFFFF40 & (lmsb - lmhb));

    glshb  = glssb - (0xFFFFFF40 & (lssb - lshb));


    /* upper accents must be at least two pixels higher than upper high chars */
    if (guab <= guht + stroke)
        guab = guht + stroke + 64;


    /* lower medium accents must be at least two pixels lower than lower medium high chars */
    if (glmab >= glmhb - stroke)
        glmab = glmhb - stroke - 64;


    /* upper medium accents must be at least two pixels higher than center high top chars */
    if (gumab <= gcht + stroke)
        gumab = gcht + stroke + 64;

    for (num = 0, i = 0; i < n_pts; i++)
    {
        /* in lower bottom zone */
        if (lhb - fuzz <= y[i] && y[i] <= lhb + fuzz)
        {
            num = add_pair(y[i], glhb, num, old_v, new_v);
        }
        /* in lower meduim bottom zone */
        else if (lmhb - fuzz <= y[i] && y[i] <= lmsb + fuzz)
        {
            if (ABS(y[i] - lmsb) < 32)
                num = add_pair(y[i], glmsb, num, old_v, new_v);
            else if (ABS(y[i] - lmhb) < 32)
                num = add_pair(y[i], glmhb, num, old_v, new_v);
        }
        /* in lower meduim accent zone */
        else if (lmab - fuzz <= y[i] && y[i] <= lmab + fuzz)
        {
            num = add_pair(y[i], glmab, num, old_v, new_v);
        }
        /* in lower short bottom zone */
        else if (lshb - fuzz <= y[i] && y[i] <= lssb + fuzz)
        {
            if (ABS(y[i] - lssb) < 32)
                num = add_pair(y[i], glssb, num, old_v, new_v);
            else if (ABS(y[i] - lshb) < 32)
                num = add_pair(y[i], glshb, num, old_v, new_v);
        }
        /* in the numerals lower bottom zone */
        else if (unb - fuzz <= y[i] && y[i] <= lnb + fuzz)
        {
            if (ABS(y[i] - lnb) < 32)
                num = add_pair(y[i], glnb, num, old_v, new_v);
            else if (ABS(y[i] - unb) < 32)
                num = add_pair(y[i], gunb, num, old_v, new_v);
        }
        /* in upper accent zone */
        else if (uab - fuzz <= y[i] && y[i] <= uab + fuzz)
        {
            num = add_pair(y[i], guab, num, old_v, new_v);
        }
        /* in upper high zone */
        else if (uht - fuzz <= y[i] && y[i] <= uht + fuzz)
        {
            num = add_pair(y[i], guht, num, old_v, new_v);
        }
        /* in upper high numeral zone */
        else if (lnt - fuzz <= y[i] && y[i] <= unt + fuzz)
        {
            if (ABS(y[i] - lnt) < 32)
                num = add_pair(y[i], glnt, num, old_v, new_v);
            else if (ABS(y[i] - unt) < 32)
                num = add_pair(y[i], gunt, num, old_v, new_v);
        }
        else if (umab - fuzz <= y[i] && y[i] <= umab + fuzz)
        {
            num = add_pair(y[i], gumab, num, old_v, new_v);
        }
        /* in center xone */
        else if (cst - fuzz <= y[i] && y[i] <= cht + fuzz)
        {
            if (ABS(y[i] - cst) < 32)
                num = add_pair(y[i], gcst, num, old_v, new_v);
            else if (ABS(y[i] - cht) < 32)
                num = add_pair(y[i], gcht, num, old_v, new_v);
        }
        /* in base line zone */
        else if (uhb - fuzz <= y[i] && y[i] <= uhb + fuzz)
        {
            num = add_pair(y[i], guhb, num, old_v, new_v);
        }

    } /* for num=0,i=sc... */

    /* (maybe) add ymax, ymin */
    /* find extrema */
    set_bbox_range(n_pts, y, &lo_y, &hi_y);
    if (int_sw)
    {
        num = add_pair(lo_y, odd_round(lo_y, y_odd), num, old_v, new_v);
        num = add_pair(hi_y, odd_round(hi_y, y_odd), num, old_v, new_v);
    }
    else
    {
        num = add_pair(lo_y, atcg(lo_y, y_sw), num, old_v, new_v);
        num = add_pair(hi_y, atcg(hi_y, y_sw), num, old_v, new_v);
    }
    /* now interpolate other points wrt these */
    sort_hints(num, old_v, new_v);
    for (i = 0; i < n_pts; i++)
    {
        y[i] = interp(y[i], num, old_v, new_v);
    }
    set_bbox_range(n_pts, y, &lo_y, &hi_y);

    /* now move interesting points to the grid, interpolate the rest */

    /** first for X **/
    for (num = 0, i = 0; i < n_pts; i++)
    {
        /* select the interesting points */
        if (x[i] == x[i + 1])
        {
            F26DOT6 z;
            /* add a new interesting point */
            if (int_sw)
                z = odd_round(x[i], x_odd);
            else
                z = atcg(x[i], x_sw);
            num = add_pair(x[i], z, num, old_v, new_v);
        }
    }

    /* (maybe) add Xmax, Xmin */
    set_bbox_range(n_pts, x, &lo_x, &hi_x);
    if (int_sw)
    {
        num = add_pair(lo_x, odd_round(lo_x, x_odd), num, old_v, new_v);
        num = add_pair(hi_x, odd_round(hi_x, x_odd), num, old_v, new_v);
    }
    else
    {
        num = add_pair(lo_x, atcg(lo_x, x_sw), num, old_v, new_v);
        num = add_pair(hi_x, atcg(hi_x, x_sw), num, old_v, new_v);
    }

    /* now interpolate other points wrt these */
    sort_hints(num, old_v, new_v);
    for (i = 0; i < n_pts; i++)
    {
        x[i] = interp(x[i], num, old_v, new_v);
    }

    /** now for Y **/
    for (num = 0, i = 0; i < n_pts; i++)
    {
        /* select the interesting points */
        if (y[i] == y[i + 1])
        {
            F26DOT6 z;
            if (int_sw)
                z = odd_round(y[i], y_odd);
            else
                z = atcg(y[i], y_sw);
            num = add_pair(y[i], z, num, old_v, new_v);
        }
    }

    /* (maybe) add Ymax, Ymin */
    set_bbox_range(n_pts, y, &lo_y, &hi_y);
    if (int_sw)
    {
        num = add_pair(lo_y, odd_round(lo_y, y_odd), num, old_v, new_v);
        num = add_pair(hi_y, odd_round(hi_y, y_odd), num, old_v, new_v);
    }
    else
    {
        num = add_pair(lo_y, atcg(lo_y, y_sw), num, old_v, new_v);
        num = add_pair(hi_y, atcg(hi_y, y_sw), num, old_v, new_v);
    }

    /* now interpolate other points wrt these */
    sort_hints(num, old_v, new_v);
    for (i = 0; i < n_pts; i++)
    {
        y[i] = interp(y[i], num, old_v, new_v);
    }

    return;
}

#endif /* FS_STIK */
